// Function to find profile URLs from LinkedIn Sales Navigator search pages
// This function will be injected into the page via chrome.scripting.executeScript
function findProfileUrlsFromSalesNavigator() {
    console.log('Starting to find URLs from LinkedIn Sales Navigator search page');

    // Implement a specialized scroll function for Sales Navigator pages
    function scrollSalesNavigatorPage() {
        return new Promise((resolve) => {
            console.log('Scrolling Sales Navigator search results...');

            // For Sales Navigator, we need to handle the scroll container differently
            // The scroll container has an id of "search-results-container"
            const scrollContainer = document.getElementById('search-results-container');
            if (!scrollContainer) {
                console.warn('Sales Navigator scroll container not found');
                resolve();
                return;
            }

            const maxScrolls = 30;
            let scrollCount = 0;
            let lastScrollHeight = 0;
            let sameHeightCount = 0;

            function scrollDown() {
                const currentScrollHeight = scrollContainer.scrollHeight;
                // Scroll down by 800px increments
                scrollContainer.scrollTop += 800;

                console.log(`Sales Navigator scroll ${scrollCount + 1}/${maxScrolls}, container height: ${currentScrollHeight}px`);

                setTimeout(() => {
                    scrollCount++;

                    // Check if we've reached the bottom of the container
                    const isAtBottom = scrollContainer.scrollTop + scrollContainer.clientHeight >= scrollContainer.scrollHeight - 100;

                    // Track if height hasn't changed to detect when all content is loaded
                    if (currentScrollHeight === lastScrollHeight) {
                        sameHeightCount++;
                    } else {
                        sameHeightCount = 0;
                        lastScrollHeight = currentScrollHeight;
                    }

                    // If we've scrolled enough times, hit the bottom, or height hasn't changed for several scrolls
                    if (scrollCount >= maxScrolls || isAtBottom || sameHeightCount >= 5) {
                        console.log(`Finished scrolling Sales Navigator after ${scrollCount} scrolls. At bottom: ${isAtBottom}`);

                        // Final scroll to ensure everything is loaded
                        scrollContainer.scrollTop = scrollContainer.scrollHeight;

                        // Wait a bit before finding
                        setTimeout(() => {
                            scrollContainer.scrollTop = 0; // Scroll back to top
                            resolve();
                        }, 2000);
                    } else {
                        scrollDown();
                    }
                }, 2000); // 2 second delay between scrolls for Sales Navigator
            }

            // Start scrolling
            scrollDown();
        });
    }

    // Function to find profile URLs from Sales Navigator page
    function getSalesNavigatorProfileUrls() {
        const profileLinks = [];
        const seenUrls = new Set(); // Track seen URLs to avoid duplicates

        // Process a Sales Navigator lead URL to convert it to a regular LinkedIn profile URL
        function processSalesNavigatorUrl(url) {
            if (!url) return null;

            // Check if it's a Sales Navigator lead URL (e.g., /sales/lead/ACwAAB4CBcIBdB1TPCqsMefq_rBlBnDPCMPSO5M)
            if (url.includes('/sales/lead/')) {
                // Extract the ID - we want the part after /sales/lead/ and before any comma or other parameter
                const match = url.match(/\/sales\/lead\/([A-Za-z0-9_-]+)(?:,|\?|&|$)/);
                if (match && match[1]) {
                    const profileId = match[1];
                    // Convert to regular LinkedIn profile URL format
                    const convertedUrl = `https://www.linkedin.com/in/${profileId}`;

                    // Only add if we haven't seen it before
                    if (!seenUrls.has(convertedUrl)) {
                        seenUrls.add(convertedUrl);
                        console.log(`Converted Sales Navigator URL to LinkedIn URL: ${convertedUrl}`);
                        return convertedUrl;
                    }
                }
            }

            return null;
        }

        // Method 1: Find all lead links with data attributes (from the HTML example)
        console.log('Looking for Sales Navigator lead links with data attributes...');
        const dataLeadLinks = document.querySelectorAll('a[data-lead-search-result="profile-link"], a[data-lead-search-result^="profile-link-"]');
        console.log(`Found ${dataLeadLinks.length} lead links with data attributes`);

        dataLeadLinks.forEach(link => {
            const href = link.getAttribute('href');
            if (href) {
                const cleanUrl = processSalesNavigatorUrl(href);
                if (cleanUrl) profileLinks.push(cleanUrl);
            }
        });

        // Method 2: Find all links in the search results that point to /sales/lead/
        console.log('Looking for all Sales Navigator lead links...');
        const allLinks = document.querySelectorAll('a[href*="/sales/lead/"]');
        console.log(`Found ${allLinks.length} total lead links`);

        allLinks.forEach(link => {
            const href = link.getAttribute('href');
            if (href) {
                const cleanUrl = processSalesNavigatorUrl(href);
                if (cleanUrl) profileLinks.push(cleanUrl);
            }
        });

        // Method 3: Find all ember view links that might be lead links
        console.log('Looking for ember view links...');
        const emberLinks = document.querySelectorAll('a.ember-view');
        emberLinks.forEach(link => {
            const href = link.getAttribute('href');
            if (href && href.includes('/sales/lead/')) {
                const cleanUrl = processSalesNavigatorUrl(href);
                if (cleanUrl) profileLinks.push(cleanUrl);
            }
        });

        // Method 4: Extract from page HTML using regex
        console.log('Extracting lead IDs from page HTML...');
        const pageHTML = document.documentElement.innerHTML;
        const leadIdRegex = /\/sales\/lead\/([A-Za-z0-9_-]+)(?:,|\?|&|"|'|\s)/g;
        let match;
        while ((match = leadIdRegex.exec(pageHTML)) !== null) {
            if (match && match[1]) {
                const profileId = match[1];
                const convertedUrl = `https://www.linkedin.com/in/${profileId}`;
                if (!seenUrls.has(convertedUrl)) {
                    seenUrls.add(convertedUrl);
                    profileLinks.push(convertedUrl);
                }
            }
        }

        console.log(`Total unique profile URLs finded from Sales Navigator: ${profileLinks.length}`);
        return profileLinks;
    }

    // Execute the finding process
    return new Promise(async (resolve) => {
        try {
            // First scroll to load all content
            await scrollSalesNavigatorPage();

            // Then find URLs
            const profileUrls = getSalesNavigatorProfileUrls();
            resolve(profileUrls);
        } catch (error) {
            console.error('Error finding Sales Navigator profiles:', error);
            // Try to get URLs even if scrolling fails
            const profileUrls = getSalesNavigatorProfileUrls();
            resolve(profileUrls);
        }
    });
}